<?php
/**
* 2007-2017 PrestaShop
*
* NOTICE OF LICENSE
*
* This source file is subject to the Academic Free License (AFL 3.0)
* that is bundled with this package in the file LICENSE.txt.
* It is also available through the world-wide-web at this URL:
* http://opensource.org/licenses/afl-3.0.php
* If you did not receive a copy of the license and are unable to
* obtain it through the world-wide-web, please send an email
* to license@prestashop.com so we can send you a copy immediately.
*
* DISCLAIMER
*
* Do not edit or add to this file if you wish to upgrade PrestaShop to newer
* versions in the future. If you wish to customize PrestaShop for your
* needs please refer to http://www.prestashop.com for more information.
*
*  @author    Cedric Fontaine <contact@prestashop.com>
*  @copyright 2007-2017 PrestaShop SA
*  @license   http://opensource.org/licenses/afl-3.0.php  Academic Free License (AFL 3.0)
*  International Registered Trademark & Property of PrestaShop SA
*/

if (!defined('_PS_VERSION_')) {
    exit;
}

/**
 * Class bluesnap
 *
 *
 * Module class
 */
class Bluesnap extends PaymentModule
{
    const PREFIX = 'BLUESNAP_';
    const SANDBOX_CHECKOUT_URL = 'https://sandbox.bluesnap.com/services/';
    const CHECKOUT_URL = 'https://ws.bluesnap.com/services/';
    const LOG_FILE = 'log/bluesnap.log';
    const EX_RATE_DEFAULT_SUM = 10000;

    public $bluesnap_url='';
    public $bluesnap_prefix='';
    private static $locally_supported = array('USD', 'EUR', 'GBP', 'CAD', 'JPY', 'CHF', 'ARS', 'AUD', 'BRL',
        'CLP', 'COP', 'HKD', 'HUF', 'INR', 'ILS', 'MYR', 'MXN', 'NZD', 'NOK', 'PLN', 'RUB', 'SAR', 'SGD',
        'ZAR', 'SEK', 'PEN', 'AED', 'AFN', 'ALL', 'AMD', 'ANG', 'AWG', 'BAM', 'BBD', 'BGN', 'BHD', 'BMD', 'BND', 'BOB', 'BSD',
        'BWP', 'BYR', 'CNY', 'CRC', 'CZK', 'DKK', 'DOP', 'DZD', 'EGP', 'FJD', 'GEL', 'GIP', 'GTQ', 'HRK', 'IDR', 'ISK', 'JMD',
        'JOD', 'KES', 'KHR', 'KRW', 'KWD', 'KYD', 'KZT', 'LBP', 'LKR', 'MAD', 'MDL', 'MKD', 'MRO', 'MUR', 'MWK', 'NAD', 'NGN',
        'NPR', 'OMR', 'PAB', 'PGK', 'PHP', 'PKR', 'QAR', 'RON', 'RSD', 'SCR', 'SDG', 'THB', 'TND', 'TRY', 'TTD', 'TWD', 'TZS',
        'UAH', 'UYU', 'UZS', 'VEF', 'VND', 'XCD', 'XOF');

    /**
     * hooks uses by module
     *
     * @var array
     */
    protected $hooks = array(
        'displayHeader',
        'payment',
        'adminOrder',
        'BackOfficeHeader',
        'displayOrderConfirmation',
        'actionObjectCurrencyUpdateBefore',
    );

    protected $html = '';

    /**
     * module settings
     *
     * @var array
     */
    protected $module_params = array(
        'USER' => '',
        'PSWD' => '',
        'SANDBOX_USER' => '',
        'SANDBOX_PSWD' => '',
        'SANDBOX' => 0,
        'SOFTDESCRIPTOR' => '',
        'API_DEBUG_MODE' => 1,
        'USE_BS_EXCHANGE' => 1,
        'VAULTED' => 1,
        'WEBHOOK_TOKEN' => '',
    );

    /**
     * create module object
     */
    public function __construct()
    {
        $this->name = 'bluesnap';
        $this->tab = 'payments_gateways';
        $this->version = '2.0.27';
        $this->author = 'Cedric Fontaine';
        $this->need_instance = 1;
        $this->is_configurable = 1;
        $this->bootstrap = true;
        $this->module_key = '76274b81856343250e6e045d5e4d2ca8';

        parent::__construct();

        $this->ps_versions_compliancy = array('min' => '1.5.0', 'max' => '1.7');
        $this->displayName = $this->l('BlueSnap Credit/Debit Card Payments');
        $this->description = $this->l('Accept online payments easily and securely with a smarter payment gateway.
				BlueSnap has helped over 5,000 merchants convert more shoppers to buyers worldwide.');

        if ($this->getConfig('SANDBOX')) {
            $this->bluesnap_url = self::SANDBOX_CHECKOUT_URL;
            $this->bluesnap_prefix = '999';
        } else {
            $this->bluesnap_url = self::CHECKOUT_URL;
        }

        if (!$this->getConfig('WEBHOOK_TOKEN')) {
            self::setConfig('WEBHOOK_TOKEN', md5(Tools::passwdGen()));
        }
    }

    /**
     * install module, register hooks, set default config values
     *
     * @return bool
     */
    public function install()
    {
        if (parent::install()) {
            foreach ($this->hooks as $hook) {
                if (!$this->registerHook($hook)) {
                    return false;
                }
            }

            if (!$this->installConfiguration()) {
                return false;
            }

            if (!function_exists('curl_version')) {
                $this->_errors[] = $this->l('Unable to install the module (CURL isn\'t installed).');
                return false;
            }

            return true;
        }

        return false;
    }

    /**
     * set default config values
     *
     * @return bool
     */
    public function installConfiguration()
    {
        foreach ($this->module_params as $param => $value) {
            if (!self::setConfig($param, $value)) {
                return false;
            }
        }

        if (!Db::getInstance()->Execute('CREATE TABLE IF NOT EXISTS `'._DB_PREFIX_.'bluesnap_order` (
                `id_bluesnap_order` int(11) unsigned NOT NULL AUTO_INCREMENT,
                `id_cart` int(11) unsigned NOT NULL,
                `bluesnap_reference` int(11) NOT NULL,
                `refund_amount` decimal(10,2) NOT NULL,
                `refunded` tinyint(1) NOT NULL,
                PRIMARY KEY (`id_bluesnap_order`)
            ) ENGINE= '._MYSQL_ENGINE_.' DEFAULT CHARSET=utf8')) {
            return false;
        }


        if (!Db::getInstance()->Execute('CREATE TABLE IF NOT EXISTS `'._DB_PREFIX_.'bluesnap_customer` (
                `id_bluesnap_customer` int(10) unsigned NOT NULL,
                `id_customer` int(10) unsigned NOT NULL,
                PRIMARY KEY (`id_bluesnap_customer`)
            ) ENGINE= '._MYSQL_ENGINE_.' DEFAULT CHARSET=utf8')) {
            return false;
        }

        return true;
    }

    /**
     * uninstall module
     *
     * @return bool
     */
    public function uninstall()
    {
        if (parent::uninstall()) {
            Db::getInstance()->Execute('DROP TABLE `'._DB_PREFIX_.'bluesnap_order`') && Db::getInstance()->Execute('DROP TABLE `'._DB_PREFIX_.'bluesnap_customer`');
            foreach ($this->hooks as $hook) {
                if (!$this->unregisterHook($hook)) {
                    return false;
                }
            }
        }

        return true;
    }

    /**
     * truncate bluesnap table in production mode
     *
     * 
     */
    public function moveToProduction()
    {
        Db::getInstance()->execute('TRUNCATE TABLE '._DB_PREFIX_.'bluesnap_order');
        Db::getInstance()->execute('TRUNCATE TABLE '._DB_PREFIX_.'bluesnap_customer');
    }

    /**
     * Return server path for file
     *
     * @param string $file
     * @return string
     */
    public function getDir($file = '')
    {
        return _PS_MODULE_DIR_.$this->name.DIRECTORY_SEPARATOR.$file;
    }

    /**
     * return correct path for .tpl file
     *
     * @param $area
     * @param $file
     * @return string
     */
    public function getTemplate($area, $file)
    {
        return 'views/templates/'.$area.'/'.$file;
    }

    /**
     * alias for Configuration::get()
     *
     * @param $name
     * @return mixed
     */
    public static function getConfig($name)
    {
        return Configuration::get(Tools::strtoupper(self::PREFIX.$name));
    }

    /**
     * alias for Configuration::get()
     *
     * @param $name
     * @return mixed
     */
    public function getUserConfig($name)
    {
        if ($this->getConfig('SANDBOX') && ($name=='user' || $name=='pswd')) {
            return Configuration::get(Tools::strtoupper(self::PREFIX.'sandbox_'.$name));
        } else {
            return Configuration::get(Tools::strtoupper(self::PREFIX.$name));
        }
    }


    /**
     * alias for Configuration::updateValue()
     *
     * @param $name
     * @param $value
     * @return mixed
     */
    public static function setConfig($name, $value)
    {
        return Configuration::updateValue(Tools::strtoupper(self::PREFIX.$name), $value);
    }

    /**
     * return html with configuration
     *
     * @return string
     */
    public function getContent()
    {
        $this->postProcess();
        $helper = $this->initForm();
        foreach ($this->fields_form as $field_form) {
            foreach ($field_form['form']['input'] as $input) {
                $helper->fields_value[$input['name']] = $this->getConfig(Tools::strtoupper($input['name']));
            }
        }

        $webhookurl = (Configuration::get('PS_SSL_ENABLED') ? 'https://' : 'http://').$_SERVER['HTTP_HOST'].__PS_BASE_URI__.''.(version_compare(_PS_VERSION_, '1.5', '>')?'index.php?process=webhook&fc=module&module=bluesnap&controller=default&':'modules/bluesnap/webhooks.php?').'token='.Tools::safeOutput($this->getConfig('WEBHOOK_TOKEN'));

        $this->smarty->assign(array(
            'webhook' => $webhookurl,
        ));

        $this->html .= $this->generateBrandbook();
        $this->html .= $helper->generateForm($this->fields_form);

        return $this->html;
    }

    public function generateBrandbook()
    {
        $this->smarty->assign(array(
            'bluesnap_img_url' => __PS_BASE_URI__.'modules/bluesnap/',
        ));

        return $this->display(__FILE__, $this->getTemplate('admin', 'brandbook.tpl'));
    }

    /**
     * helper with configuration
     *
     * @return HelperForm
     */
    private function initForm()
    {
        $helper = new HelperForm();
        $helper->module = $this;
        $helper->name_controller = $this->name;
        $helper->identifier = $this->identifier;
        $helper->token = Tools::getAdminTokenLite('AdminModules');
        $helper->currentIndex = AdminController::$currentIndex.'&configure='.$this->name;
        $helper->toolbar_scroll = true;
        $helper->toolbar_btn = $this->initToolbar();
        $helper->title = $this->displayName;
        $helper->submit_action = 'submitUpdate';

        $this->fields_form[0]['form'] = array(
            'tinymce' => true,
            'legend' => array('title' => $this->l('BlueSnap API'), 'image' => $this->_path.
                'logo.gif'),
            'submit' => array(
                'name' => 'submitUpdate',
                'title' => $this->l('   Save   ')
            ),
            'input' => array(
                array(
                    'type' => 'switch',
                    'values' => array(
                        array('label' => $this->l('Yes'), 'value' => 1, 'id' => 'sandbox_on'),
                        array('label' => $this->l('No'), 'value' => 0, 'id' => 'sandbox_off'),
                    ),
                    'is_bool' => true,
                    'class' => 't',
                    'label' => $this->l('Sandbox mode'),
                    'name' => 'sandbox',
                ),
                array(
                    'type' => 'text',
                    'label' => $this->l('Sandbox username'),
                    'name' => 'sandbox_user',
                    'size' => 64,
                ),
                array(
                    'type' => 'password',
                    'label' => $this->l('Sandbox password'),
                    'name' => 'sandbox_pswd',
                    'size' => 64,
                    'desc' => $this->l('Leave this field blank if you do not want to change your password.')
                ),
                array(
                    'type' => 'text',
                    'label' => $this->l('Username'),
                    'name' => 'user',
                    'size' => 64,
                ),
                array(
                    'type' => 'password',
                    'label' => $this->l('Password'),
                    'name' => 'pswd',
                    'size' => 64,
                    'desc' => $this->l('Leave this field blank if you do not want to change your password.')
                ),

            ),
        );

        $this->fields_form[1]['form'] = array(
            'tinymce' => true,
            'legend' => array('title' => $this->l('BlueSnap General Settings'), 'image' => $this->_path.
                'logo.gif'),
            'submit' => array(
                'name' => 'submitUpdate',
                'title' => $this->l('   Save   ')
            ),
            'input' => array(
                array(
                    'type' => 'text',
                    'label' => $this->l('Statement Soft Descriptor'),
                    'name' => 'softdescriptor',
                    'size' => 20,
                    'required' => 'true'
                ),
                array(
                    'type' => 'switch',
                    'values' => array(
                        array('label' => $this->l('Yes'), 'value' => 1, 'id' => 'debug_on'),
                        array('label' => $this->l('No'), 'value' => 0, 'id' => 'debug_off'),
                    ),
                    'is_bool' => true,
                    'class' => 't',
                    'label' => $this->l('Is Debug Mode Enabled'),
                    'name' => 'api_debug_mode',
                ),
                array(
                    'type' => 'switch',
                    'values' => array(
                        array('label' => $this->l('Yes'), 'value' => 1, 'id' => 'vaulted_on'),
                        array('label' => $this->l('No'), 'value' => 0, 'id' => 'vaulted_off'),
                    ),
                    'is_bool' => true,
                    'class' => 't',
                    'label' => $this->l('Enable Vaulted Shoppers'),
                    'name' => 'vaulted',
                ),
            ),
        );

        $this->fields_form[2]['form'] = array(
            'tinymce' => true,
            'legend' => array('title' => $this->l('BlueSnap Exchange Rate API'), 'image' => $this->_path.
                'logo.gif'),
            'submit' => array(
                'name' => 'submitUpdate',
                'title' => $this->l('   Save   ')
            ),
            'input' => array(
                array(
                    'type' => 'switch',
                    'values' => array(
                        array('label' => $this->l('Yes'), 'value' => 1, 'id' => 'bs_exchange_on'),
                        array('label' => $this->l('No'), 'value' => 0, 'id' => 'bs_exchange_off'),
                    ),
                    'is_bool' => true,
                    'class' => 't',
                    'label' => $this->l('Use BlueSnap Exchange Rate converter'),
                    'name' => 'use_bs_exchange',
                ),
            ),
        );

        return $helper;
    }

    public function refreshCurrencies()
    {
        if ($this->getConfig('SANDBOX')) {
            $this->bluesnap_url = self::SANDBOX_CHECKOUT_URL;
        } else {
            $this->bluesnap_url = self::CHECKOUT_URL;
        }

        if ($this->getUserConfig('user')=='' || $this->getUserConfig('pswd')=='') {
            return false;
        }

        // get shop default currency
        if (!$default_currency = Currency::getDefaultCurrency()) {
            return Tools::displayError('No default currency');
        }

        $default_iso_code = $default_currency->iso_code;
        $currencies = Currency::getCurrencies(true, false, true);

        /* @var $currency Currency */
        foreach ($currencies as $currency) {
            if ($currency->id != $default_currency->id) {
                if ($conversion_rate = $this->getCurrencyRate($default_iso_code, $currency->iso_code)) {
                    $currency->conversion_rate = $conversion_rate;
                    $currency->update();
                }
            }
        }

        return null;
    }

    /**
     * PrestaShop way save button
     *
     * @return mixed
     */
    private function initToolbar()
    {
        $toolbar_btn = array();
        $toolbar_btn['save'] = array('href' => '#', 'desc' => $this->l('Save'));
        return $toolbar_btn;
    }

    /**
     * save configuration values
     */
    protected function postProcess()
    {
        if (Tools::isSubmit('submitUpdate')) {
            $data = $_POST;
            $errors = array();
            if (is_array($data)) {
                if (!Tools::getValue('softdescriptor') || Tools::strlen(Tools::getValue('softdescriptor'))>20) {
                    $errors[] = $this->l('Soft descriptor is invalid.');
                } else {
                    foreach ($data as $key => $value) {
                        if (in_array($key, array('sandbox_pswd', 'pswd')) && empty($value)) {
                            continue;
                        }
                        if ($key == 'sandbox' && $value==0 && $this->getConfig('SANDBOX')==1)
                             $this->moveToProduction();
                        self::setConfig($key, $value);
                    }
                }
            }
            if (sizeof($errors)==0 && $this->getConfig('USE_BS_EXCHANGE')) {
                $this->refreshCurrencies();
            }
            if (sizeof($errors)>0) {
                $this->smarty->assign('errors', $errors);
            } else {
                Tools::redirectAdmin('index.php?tab=AdminModules&conf=4&configure='.$this->name.
                '&token='.Tools::getAdminToken('AdminModules'.
                (int)Tab::getIdFromClassName('AdminModules').(int)$this->context->employee->id));
            }
        }
    }

    /**
     * include css file in frontend
     *
     * @param $params
     */
    public function hookHeader()
    {
        /* Continue only if we are in the checkout process */
        if (Tools::getValue('controller') != 'order-opc' && (!($_SERVER['PHP_SELF'] == __PS_BASE_URI__.'order.php' || $_SERVER['PHP_SELF'] == __PS_BASE_URI__.'order-opc.php' || Tools::getValue('controller') == 'order' || Tools::getValue('controller') == 'orderopc' || Tools::getValue('step') == 3))) {
            return;
        }

        if (!Configuration::get('PS_SSL_ENABLED') && !$this->getConfig('SANDBOX')) {
            return;
        }

        // OK PASSÉ ICI
        // PENSÉ À ACTIVER LE CSS
        $this->context->controller->addJS($this->bluesnap_url.'/hosted-payment-fields/v1.0/bluesnap.hpf.mini.js');
        $this->context->controller->addCSS(($this->_path).'views/css/front.css', 'all');
    }

    /**
     * show module on payment step
     *
     * @param $params
     * @return mixed
     */
    public function hookPayment()
    {
        $current_currency_code = $this->context->currency->iso_code;

        if (!$this->active || !$this->isLocallySupported($current_currency_code)) {
            return;
        }

        if (!Configuration::get('PS_SSL_ENABLED') && !$this->getConfig('SANDBOX')) {
            return;
        }

        $userpass =  ($this->getUserConfig('user').':'.$this->getUserConfig('pswd'));

        $existingCustomer = array();
        $customer = new Customer((int)$this->context->cart->id_customer);
        // Check token
        if (!$customer->isGuest() && $this->getConfig('VAULTED')==1) {
            if ($row = Db::getInstance()->getRow(
                'SELECT *
				FROM `'._DB_PREFIX_.'bluesnap_customer`
				WHERE `id_customer` = '.$this->bluesnap_prefix.$this->context->cart->id_customer
            )) {
                $url = $this->bluesnap_url.'2/vaulted-shoppers/'.$row['id_bluesnap_customer'];
                $curl = curl_init();
                curl_setopt($curl, CURLOPT_URL, $url);
                curl_setopt($curl, CURLOPT_RETURNTRANSFER, true);
                curl_setopt($curl, CURLOPT_FOLLOWLOCATION, true);
                curl_setopt($curl, CURLOPT_HTTPHEADER, array('Content-Type: application/json','Accept: application/json'));
                curl_setopt($curl, CURLOPT_MAXREDIRS, 1);
                curl_setopt($curl, CURLOPT_HTTPAUTH, CURLAUTH_BASIC);
                curl_setopt($curl, CURLOPT_USERPWD, $userpass);
                curl_setopt($curl, CURLOPT_CONNECTTIMEOUT, 30);
                curl_setopt($curl, CURLOPT_TIMEOUT, 30);

                $curl_response = curl_exec($curl); // Execute REST Request
                $this->log("GET : ".$url."\nRESPONSE:\n".print_r($curl_response, true));

                if (!curl_errno($curl) && curl_getinfo($curl, CURLINFO_HTTP_CODE)=='200') {
                    $response = Tools::jsonDecode($curl_response);
                    $payments = $response->paymentSources->creditCardInfo;
                    foreach ($payments as $payment) {
                        /*if ($payment->creditCard->expirationYear >= date('Y')) {*/
                            $ccard = $payment->creditCard;
                        $existingCustomer[] = array('uid'=>$ccard->cardLastFourDigits."#".$ccard->cardType."#".(property_exists('ccard', 'cardSubType')? $ccard->cardSubType : "")."#".$ccard->expirationMonth."#".$ccard->expirationYear,'ccdata'=>$ccard->cardType." ".$ccard->cardLastFourDigits." ".$ccard->expirationMonth."/".$ccard->expirationYear);
                        /*}*/
                    }
                } else {
                    $this->log('Unable to get vaulted shoppers : '.curl_errno($curl));
                }
            }
        }

        $this->smarty->assign('existingCustomer', $existingCustomer);


        // Avoid creating more than one token for 15mn
        $bscookie = new Cookie('Bluesnap');
        if ($bscookie->token && $bscookie->token!='') {
            $token = $bscookie->token;
        } else {
            $url = $this->bluesnap_url.'2/payment-fields-tokens';

            $curl = curl_init();
            curl_setopt($curl, CURLOPT_URL, $url);
            curl_setopt($curl, CURLOPT_RETURNTRANSFER, true);
            curl_setopt($curl, CURLOPT_FOLLOWLOCATION, true);
            curl_setopt($curl, CURLOPT_HEADER, true);
            curl_setopt($curl, CURLOPT_MAXREDIRS, 1);
            curl_setopt($curl, CURLOPT_HTTPAUTH, CURLAUTH_BASIC);
            curl_setopt($curl, CURLOPT_USERPWD, $userpass);
            curl_setopt($curl, CURLOPT_CONNECTTIMEOUT, 30);
            curl_setopt($curl, CURLOPT_TIMEOUT, 30);
            curl_setopt($curl, CURLOPT_POST, true);
            $curl_response = curl_exec($curl); // Execute REST Request
            $this->log("POST : ".$url."\nRESPONSE:\n".print_r($curl_response, true));

            if (!curl_errno($curl) && curl_getinfo($curl, CURLINFO_HTTP_CODE)=='201') {
                $curl_info = curl_getinfo($curl);
                $headers = Tools::substr($curl_response, 0, $curl_info["header_size"]); //split out header
                preg_match("!\r\n(?:Location|URI): *(.*?) *\r\n!", $headers, $matches);
                $token = str_replace($url.'/', "", $matches[1]);
                if ($token == '') {
                    $this->log('No token : '.$curl_response);
                    return false;
                }
            } else {
                $this->log('Unable to get token : '.curl_errno($curl));
                return false;
            }
            $bscookie->setExpire(time() + 15 * 60); // 15minutes
            $bscookie->token = $token;
            $bscookie->write();
        }

        if ($bscookie->error != '') {
            $bserror = $bscookie->error;
            $bscookie->__unset('error');
            $bscookie->write();
        } else {
            $bserror = "";
        }

        $cookie = new Cookie('psAdmin', '', (int)Configuration::get('PS_COOKIE_LIFETIME_BO'));
        $isLoggedAdmin = isset($cookie->id_employee) && isset($cookie->passwd) && Employee::checkPassword($cookie->id_employee, $cookie->passwd);
        if ($isLoggedAdmin) {
            $bluesnap_uuid = md5(_COOKIE_KEY_.$cookie->id_employee);
        } else {
            $bluesnap_uuid = md5(_COOKIE_KEY_.$this->context->cart->id_guest);
        }

        $this->smarty->assign(array(
            'bluesnap_token' => $token,
            'bluesnap_uuid' => $bluesnap_uuid,
            'bluesnap_url' => $this->bluesnap_url,
            'bluesnap_sandbox' => $this->getConfig('SANDBOX'),
            'bluesnap_error' => $bserror,
            'bluesnap_isadmin' => (bool)$isLoggedAdmin,
        ));

        return $this->display(__FILE__, $this->getTemplate('front', 'payment.tpl'));
    }

    public function hookAdminOrder()
    {
        if (Tools::isSubmit('id_order')) {
            $order_obj = new Order(Tools::getValue('id_order'));
            
            $bluesnap_info = Db::getInstance()->getRow('SELECT * FROM `'._DB_PREFIX_.'bluesnap_order` WHERE `id_cart` = \''.(int)$order_obj->id_cart.'\'');
            if ($bluesnap_info && !empty($bluesnap_info)) {
                if ($bluesnap_info['refund_amount'] != $order_obj->total_paid) {
                    $bluesnap_info['refunded'] = 0;
                }

                $this->context->smarty->assign(array(
                    'bluesnap_amount' => $order_obj->total_paid-$bluesnap_info['refund_amount'],
                    'bluesnap_reference_number' => $bluesnap_info['bluesnap_reference'],
                    'bluesnap_refund_amount' => $bluesnap_info['refund_amount'],
                    'bluesnap_refunded' => (int)$bluesnap_info['refunded'],
                    'id_currency' => (int)$order_obj->id_currency
                ));

                return $this->display(__FILE__, $this->getTemplate('admin', 'order.tpl'));
            }
        }
    }

    public function hookBackOfficeHeader()
    {
        $reference_number = Tools::getValue('bluesnap_reference_number');
        $amount = Tools::getValue('bluesnap_amount');
        $id_order = (int)Tools::getValue('id_order');
        $bluesnap_error = false;
        $order_obj = new Order($id_order);
        $bluesnap_info = Db::getInstance()->getRow('SELECT * FROM `'._DB_PREFIX_.'bluesnap_order` WHERE `id_cart` = \''.(int)$order_obj->id_cart.'\'');
        if (!$bluesnap_info || empty($bluesnap_info)) {
            return false;
        }
        if ($amount > ($order_obj->total_paid-$bluesnap_info['refund_amount'])) {
            $bluesnap_error = $this->l('Amount could not be higher than total paid');
        } elseif ($reference_number && $id_order) {
            $id_order_state = Configuration::get('PS_OS_REFUND');
            $template_vars = array();

            $reason=urlencode('BackOffice employee id '.$this->context->employee->id);

            if ($amount == $order_obj->total_paid) {
                $url = $this->bluesnap_url.'2/transactions/'.(int)$reference_number.'/refund?reason='.$reason;
            } else {
                $url = $this->bluesnap_url.'2/transactions/'.(int)$reference_number.'/refund?reason='.$reason.'&amount='.$amount;
            }

            $userpass =  ($this->getUserConfig('user').':'.$this->getUserConfig('pswd'));
            $curl = curl_init();
            curl_setopt($curl, CURLOPT_URL, $url);
            curl_setopt($curl, CURLOPT_RETURNTRANSFER, true);
            curl_setopt($curl, CURLOPT_FOLLOWLOCATION, true);
            curl_setopt($curl, CURLOPT_MAXREDIRS, 1);
            curl_setopt($curl, CURLOPT_HTTPAUTH, CURLAUTH_BASIC);
            curl_setopt($curl, CURLOPT_USERPWD, $userpass);
            curl_setopt($curl, CURLOPT_HTTPHEADER, array('Content-Type: application/json'));
            curl_setopt($curl, CURLOPT_CUSTOMREQUEST, 'PUT');
            curl_setopt($curl, CURLOPT_CONNECTTIMEOUT, 30);
            curl_setopt($curl, CURLOPT_TIMEOUT, 30);

            $curl_response = curl_exec($curl); // Execute REST Request
            $this->log("PUT : ".$url."\nRESPONSE:\nCURLINFO_HTTP_CODE".curl_getinfo($curl, CURLINFO_HTTP_CODE)."\n".print_r($curl_response, true));

            if (!curl_errno($curl) && curl_getinfo($curl, CURLINFO_HTTP_CODE)=='204') {
                $orders_collection = Order::getByReference($order_obj->reference);
                foreach ($orders_collection->getResults() as $order) {
                    // Set new order state
                    $new_history = new OrderHistory();
                    $new_history->id_order = (int)$order->id;
                    $new_history->changeIdOrderState((int)$id_order_state, $order, true);
                    // Save all changes
                    if ($new_history->addWithemail(true, $template_vars)) {
                        Db::getInstance()->Execute('
							UPDATE '._DB_PREFIX_.'bluesnap_order SET `refunded`=1, `refund_amount`=`refund_amount`+'.pSQL($amount).' WHERE `bluesnap_reference`=\''.pSQL($reference_number).'\';');
                    }
                }
            } else {
                $message = '';
                if (!curl_errno($curl) && curl_getinfo($curl, CURLINFO_HTTP_CODE)==400) {
                    $response = Tools::jsonDecode($curl_response);
                    foreach ($response->message as $messages) {
                        $message.=$messages->description.' ';
                    }
                }
                $bluesnap_error = $this->l('An error has occurred.').' '.$message.' '.$this->l('Please contact BlueSnap support (merchants@bluesnap.com) for further assistance');
            }
        }
        $this->context->smarty->assign('bluesnap_error', $bluesnap_error);
    }

    public function hookActionObjectCurrencyUpdateBefore($params)
    {
        if ($this->getUserConfig('user')=='' || $this->getUserConfig('pswd')=='') {
            return false;
        }

        // get shop default currency
        if (!$default_currency = Currency::getDefaultCurrency()) {
            return Tools::displayError('No default currency');
        }

        // do not apply changes if currency is updated mannualy or BlueSnap EX Rates API is disabled
        if (!Tools::getIsset('submitUpdate') && !Tools::getIsset('submitAddcurrency') && $this->getConfig('USE_BS_EXCHANGE')) {
            $currency = $params['object'];
            $default_iso_code = $default_currency->iso_code;
            $conversion_rate = $this->getCurrencyRate($default_iso_code, $currency->iso_code);

            if ($conversion_rate != null && $conversion_rate != $currency->conversion_rate) {
                $currency->conversion_rate = $conversion_rate;
            }
        }
    }

    /**
     * save log file
     *
     * @param $string
     * @param null $file
     */
    public static function log($string, $file = null)
    {
        if (self::getConfig('API_DEBUG_MODE')==0) {
            return;
        }

        if (empty($file)) {
            $file = self::LOG_FILE;
        }

        $file = dirname(__FILE__).DS.$file;

        file_put_contents($file, date('Y-m-d H:i:s')."\n".$string."\n\n", FILE_APPEND | LOCK_EX);
    }

    public function hookDisplayOrderConfirmation($params)
    {
        if (!isset($params['objOrder']) || ($params['objOrder']->module != $this->name)) {
            return false;
        }

        if (isset($params['objOrder']) && Validate::isLoadedObject($params['objOrder']) && isset($params['objOrder']->valid) && version_compare(_PS_VERSION_, '1.5', '>=') && isset($params['objOrder']->reference)) {
            $this->smarty->assign('bluesnap_order', array(
                'id' => $params['objOrder']->id,
                'reference' => $params['objOrder']->reference,
                'valid' => $params['objOrder']->valid,
                'total_to_pay' => Tools::displayPrice($params['total_to_pay'], $params['currencyObj'], false)
                )
            );
            return $this->display(__FILE__, $this->getTemplate('front', 'order-confirmation.tpl'));
        }
    }

    /**
     * Get currency exchange rate
     * @param string $from
     * @param string $to
     * @return float|null
     */
    public function getCurrencyRate($from, $to)
    {
        $amount = self::EX_RATE_DEFAULT_SUM;
        $url = $this->bluesnap_url.'2/tools/merchant-currency-convertor';
        $url .= '?'.http_build_query(array('from' => $from, 'to' => $to, 'amount' => $amount));

        $userpass =  ($this->getUserConfig('user').':'.$this->getUserConfig('pswd'));
        $curl = curl_init();
        curl_setopt($curl, CURLOPT_URL, $url);
        curl_setopt($curl, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($curl, CURLOPT_FOLLOWLOCATION, true);
        curl_setopt($curl, CURLOPT_MAXREDIRS, 1);
        curl_setopt($curl, CURLOPT_HTTPAUTH, CURLAUTH_BASIC);
        curl_setopt($curl, CURLOPT_USERPWD, $userpass);
        curl_setopt($curl, CURLOPT_CONNECTTIMEOUT, 30);
        curl_setopt($curl, CURLOPT_TIMEOUT, 30);

        $curl_response = curl_exec($curl); // Execute REST Request
        $this->log("GET : ".$url."\nRESPONSE:\n".print_r($curl_response, true));
        
        if (!curl_errno($curl) && curl_getinfo($curl, CURLINFO_HTTP_CODE)=='200') {
            // create XML object
            libxml_use_internal_errors(true);
            $response = simplexml_load_string($curl_response);

            if ($response) {
                if ($response->value) {
                    $result = ((float)$response->value) / self::EX_RATE_DEFAULT_SUM;
                } else {
                    return false;
                }
            }

            return $result;
        } else {
            $this->log('Unable to get currency conversion : '.curl_errno($curl));
            return false;
        }
    }

    /**
     * Check if currency is on a BS locally supported currencis list
     * @param string $currency_code
     * @return bool
     */
    public static function isLocallySupported($currency_code)
    {
        return in_array($currency_code, self::$locally_supported);
    }
}
