<?php
/**
 * NOTICE OF LICENSE
 *
 * This source file is subject to a commercial license from Common-Services Co., Ltd.
 * Use, copy, modification or distribution of this source file without written
 * license agreement from the SARL SMC is strictly forbidden.
 * In order to obtain a license, please contact us: contact@common-services.com
 * ...........................................................................
 * INFORMATION SUR LA LICENCE D'UTILISATION
 *
 * L'utilisation de ce fichier source est soumise a une licence commerciale
 * concedee par la societe Common-Services Co., Ltd.
 * Toute utilisation, reproduction, modification ou distribution du present
 * fichier source sans contrat de licence ecrit de la part de la Common-Services Co. Ltd. est
 * expressement interdite.
 * Pour obtenir une licence, veuillez contacter Common-Services Co., Ltd. a l'adresse: contact@common-services.com
 *
 * @author    Olivier B.
 * @copyright Copyright (c) 2011-2018 Common Services Co Ltd - 90/25 Sukhumvit 81 - 10260 Bangkok - Thailand
 * @license   Commercial license
 * @package   Amazon Market Place
 * Support by mail:  support.amazon@common-services.com
 */

class AmazonOrdersReports extends AmazonWebService
{
    const ONE_DAY = 86400;
    const LAG = 300;

    const STATUS_SUBMITTED  = '_SUBMITTED_';
    const STATUS_IN_PROGRESS  = '_IN_PROGRESS_';
    const STATUS_CANCELLED  = '_CANCELLED_';
    const STATUS_DONE  = '_DONE_';
    const STATUS_DONE_NO_DATA  = '_DONE_NO_DATA_';

    protected static $orders_fields = array(
        'order_id',
        'order_item_id',
        'purchase_date',
        'payments_date',
        'buyer_email',
        'buyer_name',
        'buyer_phone_number',
        'sku',
        'product_name',
        'quantity_purchased',
        'currency',
        'item_price',
        'item_tax',
        'shipping_price',
        'shipping_tax',
        'ship_service_level',
        'recipient_name',
        'ship_address_1',
        'ship_address_2',
        'ship_address_3',
        'ship_city',
        'ship_state',
        'ship_postal_code',
        'ship_country',
        'ship_phone_number',
        'bill_address_1',
        'bill_address_2',
        'bill_address_3',
        'bill_city',
        'bill_state',
        'bill_postal_code',
        'bill_country',
        'item_promotion_discount',
        'item_promotion_id',
        'ship_promotion_discount',
        'ship_promotion_id',
        'delivery_start_date',
        'delivery_end_date',
        'delivery_time_zone',
        'delivery_Instructions',
        'sales_channel'
    );

    protected static $save_orders_fields = array(
        'order_id',
        'purchase_date',
        'payments_date',
        'buyer_email',
        'buyer_name',
        'buyer_phone_number',
        'recipient_name',
        'ship_address_1',
        'ship_address_2',
        'ship_address_3',
        'ship_city',
        'ship_state',
        'ship_postal_code',
        'ship_country',
        'ship_phone_number',
        'bill_address_1',
        'bill_address_2',
        'bill_address_3',
        'bill_city',
        'bill_state',
        'bill_postal_code',
        'bill_country',
        'delivery_start_date',
        'delivery_end_date',
        'delivery_time_zone',
        'delivery_Instructions',
        'sales_channel'
    );

    public function parseReport($filename)
    {
        if (($result = Tools::file_get_contents($filename)) === false) {
            self::$errors[] = $error = sprintf('Unable to read input file (%s)', $this->file_orders);

            if (Amazon::$debug_mode) {
                echo '<pre>'.Amazon::LF;
                printf('%s(#%d): init(): Failed - %s', basename(__FILE__), __LINE__, $error);
                echo "</pre>".Amazon::LF;
            }
            return (false);
        }

        if ($result == null or empty($result)) {
            printf('%s(#%d): processOrders -Orders is empty !', basename(__FILE__), __LINE__);
            return (false);
        }

        $lines = explode("\n", $result);

        if (!is_array($lines) || !count($lines)) {
            printf('%s(#%d): processOrders - Orders is empty !', basename(__FILE__), __LINE__);
            return (false);
        }

        if (Amazon::$debug_mode) {
            echo "<pre>\n";
            echo str_repeat('-', 160)."\n";
            printf('Orders: %s lines'."<br />\n", count($lines));
            echo "</pre>\n";
        }

        $header = reset($lines);


        if (Amazon::$debug_mode) {
            echo "<pre>\n";
            echo str_repeat('-', 160)."\n";
            printf('Header: %s'."<br />\n", nl2br(print_r(explode("\t", $header), true)));
            echo "</pre>\n";
        }

        $count = 0;
        $orders = array();

        foreach ($lines as $line) {
            $order = array();

            if (empty($line)) {
                continue;
            }

            if ($count++ < 1) {
                continue;
            }

            $result = explode("\t", $line);

            if (count(array_keys($result)) < 4) {
                continue;
            }
            foreach (self::$orders_fields as $key => $order_field) {
                if (in_array($order_field, self::$save_orders_fields)) {
                    $order[$order_field] = $result[$key];
                }
            }
            $orders[$order['order_id']] = $order;// unduplicates records

            if (Amazon::$debug_mode) {
                echo "<pre>\n";
                echo str_repeat('-', 160)."\n";
                printf('Order: %s'."<br />\n", nl2br(print_r($order, true)));
                echo "</pre>\n";
            }
        }
        /* Array of :
            '305-0743990-0302739' =>
                array (size=27)
                  'order_id' => string '305-0743990-0302739' (length=19)
                  'purchase_date' => string '2017-06-20T12:51:03+00:00' (length=25)
                  'payments_date' => string '2017-06-20T12:51:03+00:00' (length=25)
                  'buyer_email' => string 'l6g84rjz41ywg6p@marketplace.amazon.de' (length=37)
                  'buyer_name' => string 'Kristof' (length=7)
                  'buyer_phone_number' => string '+49 123456789' (length=13)
                  'recipient_name' => string 'Kristof Dieckmann' (length=17)
                  'ship_address_1' => string 'c/o DUKAL W�SCHE GmbH' (length=23)
                  'ship_address_2' => string 'Stiergasse 10' (length=13)
                  'ship_address_3' => string '' (length=0)
                  'ship_city' => string 'Durchhausen' (length=11)
                  'ship_state' => string 'Baden-W�rttemberg' (length=19)
                  'ship_postal_code' => string '78591' (length=5)
                  'ship_country' => string 'DE' (length=2)
                  'ship_phone_number' => string '074641336' (length=9)
                  'bill_address_1' => string 'musterstr. 1' (length=12)
                  'bill_address_2' => string '' (length=0)
                  'bill_address_3' => string '' (length=0)
                  'bill_city' => string 'trossingen' (length=10)
                  'bill_state' => string '' (length=0)
                  'bill_postal_code' => string '78647' (length=5)
                  'bill_country' => string 'DE' (length=2)
                  'delivery_start_date' => string '' (length=0)
                  'delivery_end_date' => string '' (length=0)
                  'delivery_time_zone' => string '' (length=0)
                  'delivery_Instructions' => string '' (length=0)
                  'sales_channel' => string '
            ' (length=1)
        */

        return($orders);
    }


    public function getReport($reportId)
    {
        $params = array();
        $params['Version'] = '2009-01-01';
        $params['Action'] = 'GetReport';
        $params['ReportId'] = (string)$reportId;

        if (Amazon::$debug_mode) {
            echo '<pre>'.Amazon::LF;
            printf('getReport()'.Amazon::LF);
            echo '</pre>'.Amazon::LF;
        }

        $result = $this->simpleCallWS($params, 'Reports', false);

        if ($result instanceof SimpleXMLElement) {
            echo '<pre>'.Amazon::LF;
            printf('%s(#%d): getReport - An error occured: %s'.Amazon::LF, basename(__FILE__), __LINE__, print_r($result, true));
            echo '</pre>'.Amazon::LF;
            return(false);
        }

        if (empty($result)) {
            echo '<pre>'.Amazon::LF;
            printf('%s(#%d): getReport - no result'.Amazon::LF, basename(__FILE__), __LINE__);
            echo '</pre>'.Amazon::LF;
            return (false);
        }

        return ($result);
    }


    public function ackReport($reportId)
    {
        $params = array();
        $params['Version'] = '2009-01-01';
        $params['Action'] = 'UpdateReportAcknowledgements';
        $params['Acknowledged'] = 'true';
        $params['ReportIdList.Id.1'] = (string)$reportId;

        if (Amazon::$debug_mode) {
            echo '<pre>'.Amazon::LF;
            printf('getReport()'.Amazon::LF);
            echo '</pre>'.Amazon::LF;
        }

        $result = $this->simpleCallWS($params, 'Reports', true);

        if ($result instanceof SimpleXMLElement && (int)$result->UpdateReportAcknowledgementsResult->Count) {
            echo '<pre>'.Amazon::LF;
            printf('%s(#%d): getReport - successfully acknoledged'.Amazon::LF, basename(__FILE__), __LINE__);
            print_r($result);
            echo '</pre>'.Amazon::LF;
            return(true);
        } else {
            echo '<pre>'.Amazon::LF;
            printf('%s(#%d): getReport - acknowledge failed: %s'.Amazon::LF, basename(__FILE__), __LINE__, print_r($result, true));
            echo '</pre>'.Amazon::LF;
            return (false);
        }
    }

    public function reportList($last)
    {
        $params = array();
        $params['Version'] = '2009-01-01';
        $params['Action'] = 'GetReportList';
        $params['ReportTypeList.Type.1'] = '_GET_FLAT_FILE_ORDERS_DATA_';
        $params['ReportProcessingStatusList.Status.1'] = '_DONE_';
        //$params['AvailableFromDate'] = gmdate('Y-m-d\TH:i:s\Z', strtotime('now -2 hours'));
        $params['AvailableFromDate'] = gmdate('Y-m-d\TH:i:s\Z', strtotime($last));
        $params['Acknowledged'] = 'false';

        if (Amazon::$debug_mode) {
            printf('reportList()'."<br />\n");
        }

        $xml = $this->simpleCallWS($params, 'Reports');

        if (!$xml instanceof SimpleXMLElement or isset($xml->Error)) {
            echo '<pre>'.Amazon::LF;
            printf('%s(#%d): %s - reportList Failed', basename(__FILE__), __LINE__);
            echo '</pre>'.Amazon::LF;

            return (false);
        }

        if (Amazon::$debug_mode) {
            echo '<pre>'.Amazon::LF;
            printf('%s(#%d): - reportList:', basename(__FILE__), __LINE__);
            echo '</pre>'.Amazon::LF;

            echo $this->debugXML($xml);
        }

        $xml->registerXPathNamespace('xmlns', 'http://mws.amazonaws.com/doc/2009-01-01/');

        $xpath_result = $xml->xpath('//xmlns:GetReportListResponse/xmlns:GetReportListResult/xmlns:ReportInfo');

        if (Amazon::$debug_mode) {
            echo "<pre>Reports:\n";
            printf('%s(#%d): reportList result: %s', basename(__FILE__), __LINE__, nl2br(print_r($xpath_result, true)));
            echo "</pre>\n";
        }

        if (is_array($xpath_result) && count($xpath_result)) {
            if (Amazon::$debug_mode) {
                echo '<pre>'.Amazon::LF;
                printf('%s(#%d): reportList - reports: %s', basename(__FILE__), __LINE__, print_r($xpath_result, true));
                echo '</pre>'.Amazon::LF;
            }
            return($xpath_result);
        } else {
            if (Amazon::$debug_mode) {
                echo '<pre>'.Amazon::LF;
                printf('%s(#%d): reportList - no report available', basename(__FILE__), __LINE__);
                echo '</pre>'.Amazon::LF;
            }
            return(false);
        }

        return (false);
    }

    public function reportRequestList($from_date = null, $to_date = null, $report_request_id = null)
    {
        $params = array();
        $params['Version'] = '2009-01-01';
        $params['Action'] = 'GetReportRequestList';
        $params['ReportTypeList.Type.1'] = '_GET_FLAT_FILE_ORDERS_DATA_';

        if ($from_date) {
            $params['RequestedFromDate'] = date('Y-m-d\TH:i:s\+00:00', strtotime($from_date));
        }
        if ($to_date) {
            $params['RequestedToDate'] = date('Y-m-d\TH:i:s\+00:00', strtotime($to_date));
        }
        if ((int)$report_request_id && is_numeric($report_request_id)) {
            $params['ReportRequestIdList.Id.1'] = $report_request_id;
        }

        if (Amazon::$debug_mode) {
            echo '<pre>'.Amazon::LF;
            printf('%s(#%d): - reportRequestList params: %s', basename(__FILE__), __LINE__, print_r($params, true));
            echo '</pre>'.Amazon::LF;
        }

        $xml = $this->simpleCallWS($params, 'Reports');

        if (!$xml instanceof SimpleXMLElement or isset($xml->Error)) {
            echo '<pre>'.Amazon::LF;
            printf('%s(#%d): - reportRequestList Failed', basename(__FILE__), __LINE__);
            echo '</pre>'.Amazon::LF;

            return (false);
        }

        if (Amazon::$debug_mode) {
            echo '<pre>'.Amazon::LF;
            printf('%s(#%d): - reportRequestList:', basename(__FILE__), __LINE__);
            echo '</pre>'.Amazon::LF;

            echo $this->debugXML($xml);
        }

        $xml->registerXPathNamespace('xmlns', 'http://mws.amazonaws.com/doc/2009-01-01/');

        $xpath_result = $xml->xpath('//xmlns:GetReportRequestListResponse/xmlns:GetReportRequestListResult/xmlns:ReportRequestInfo');

        if (Amazon::$debug_mode) {
            echo "<pre>Reports:\n";
            printf('%s(#%d): reportRequestList result: %s', basename(__FILE__), __LINE__, nl2br(print_r($xpath_result, true)));
            echo "</pre>\n";
        }

        if (is_array($xpath_result) && !count($xpath_result)) {
            echo '<pre>'.Amazon::LF;
            printf('%s(#%d): reportRequestList - no report available', basename(__FILE__), __LINE__);
            echo '</pre>'.Amazon::LF;

            return(false);
        } else {
            foreach ($xpath_result as $key => $report_data) {
                if ($report_data instanceof SimpleXMLElement) {
                    if (!isset($report_data->ReportRequestId) || !(int)$report_data->ReportRequestId) {
                        unset($xpath_result[$key]);
                        continue;
                    }
                    if (!isset($report_data->CompletedDate) || strtotime($report_data->CompletedDate) < time() - (self::ONE_DAY * self::EXPIRES)) {
                        if (Amazon::$debug_mode) {
                            echo '<pre>'.Amazon::LF;
                            printf('%s(#%d): reportRequestList - skipping expired: %s'.Amazon::LF, basename(__FILE__), __LINE__, print_r($report_data, true));
                            echo '</pre>'.Amazon::LF;
                        }
                        unset($xpath_result[$key]);
                        continue;
                    }
                    if (Amazon::$debug_mode) {
                        echo '<pre>'.Amazon::LF;
                        printf('%s(#%d): reportRequestList - selected report: %s'.Amazon::LF, basename(__FILE__), __LINE__, print_r($report_data, true));
                        echo '</pre>'.Amazon::LF;
                    }
                }
            }
            $xpath_result = array_values(array_filter($xpath_result));

            if (!count($xpath_result)) {
                echo '<pre>'.Amazon::LF;
                printf('%s(#%d): reportRequestList - no report available', basename(__FILE__), __LINE__);
                echo '</pre>'.Amazon::LF;
                return(false);
            }
            return($xpath_result);
        }

        return (false);
    }

    public function reportRequest($date_start, $date_end = null)
    {
        $params = array();
        $params['Action'] = 'RequestReport';
        $params['ReportType'] = '_GET_FLAT_FILE_ORDERS_DATA_';
        $params['Version'] = '2009-01-01';
        $params['StartDate'] = gmdate('Y-m-d\TH:i:s\Z', strtotime($date_start));

        if ($date_end) {
            $params['EndDate'] = gmdate('Y-m-d\TH:i:s\Z', strtotime($date_end));
        }

        if (Amazon::$debug_mode) {
            echo '<pre>'.Amazon::LF;
            printf('reportRequest()'."<br />\n");
            printf('%s(#%d): reportRequest: %s'.Amazon::LF, basename(__FILE__), __LINE__, print_r($params, true));
            echo '</pre>'.Amazon::LF;
        }

        $xml = $this->simpleCallWS($params, 'Reports');

        if (!$xml instanceof SimpleXMLElement or isset($xml->Error)) {
            echo '<pre>'.Amazon::LF;
            printf('%s(#%d): reportRequest Failed: %s', basename(__FILE__), __LINE__, print_r($xml, true));
            echo '</pre>'.Amazon::LF;

            return (false);
        }

        if (Amazon::$debug_mode) {
            echo  $this->debugXML($xml);
        }

        if (!isset($xml->RequestReportResult->ReportRequestInfo->ReportProcessingStatus) || !isset($xml->RequestReportResult->ReportRequestInfo->ReportRequestId)) {
            echo '<pre>'.Amazon::LF;
            printf('%s(#%d): reportRequest Failed', basename(__FILE__), __LINE__);
            echo '</pre>'.Amazon::LF;

            return (false);
        }

        if ((string)$xml->RequestReportResult->ReportRequestInfo->ReportProcessingStatus == '_SUBMITTED_') {
            echo '<pre>'.Amazon::LF;
            printf('%s(#%d):'.Amazon::LF, basename(__FILE__), __LINE__);
            printf('%s(#%d): reportRequest Status: %s'.Amazon::LF, basename(__FILE__), __LINE__, (string)$xml->RequestReportResult->ReportRequestInfo->ReportProcessingStatus);
            printf('%s(#%d): reportRequest ReportId: %s'.Amazon::LF, basename(__FILE__), __LINE__, (string)$xml->RequestReportResult->ReportRequestInfo->ReportRequestId);
            printf('%s(#%d): reportRequest SubmittedDate: %s'.Amazon::LF, basename(__FILE__), __LINE__, (string)$xml->RequestReportResult->ReportRequestInfo->SubmittedDate);
            echo '</pre>'.Amazon::LF;

            $report_request_id = (string)$xml->RequestReportResult->ReportRequestInfo->ReportRequestId;

            return ($report_request_id);
        } else {
            echo '<pre>'.Amazon::LF;
            printf('%s(#%d): %s - reportRequest FAILED'.Amazon::LF, basename(__FILE__), __LINE__);
            echo '</pre>'.Amazon::LF;

            return (false);
        }
    }


    public function getReportScheduleList()
    {
        $params = array();
        $params['Version'] = '2009-01-01';
        $params['Action'] = 'GetReportScheduleList';
        $params['ReportTypeList.Type.1'] = '_GET_ORDERS_DATA_';

        if (Amazon::$debug_mode) {
            echo '<pre>'.Amazon::LF;
            printf('getReportScheduleList()'.Amazon::LF);
            echo "</pre>".Amazon::LF;
        }
        $report_explainations = $this->l('Here is displayed the report scheduler status, more details at the following URL');
        $url = AmazonSupport::gethreflink(AmazonSupport::TUTORIAL_ORDERS_REPORT);

        self::$messages[] = sprintf('%s: %s', $report_explainations, $url);



        $xml = $this->simpleCallWS($params, 'Reports');

        if (Amazon::$debug_mode) {
            echo '<pre>'.Amazon::LF;
            printf('getReportScheduleList() - result: %s'.Amazon::LF, print_r($xml, true));
            echo "</pre>".Amazon::LF;
        }

        if (!$xml instanceof SimpleXMLElement or isset($xml->Error)) {
            $error = 'API Error';

            if (Amazon::$debug_mode) {
                echo '<pre>'.Amazon::LF;
                printf(sprintf('%s(#%d): GetReportScheduleListResponse Failed - %s', basename(__FILE__), __LINE__), $error);
                echo "</pre>".Amazon::LF;
            }
            die;
        }

        if (Amazon::$debug_mode) {
            echo '<pre>'.Amazon::LF;
            printf('getReportScheduleList() - report'.Amazon::LF);
            echo "</pre>".Amazon::LF;
            echo $this->debugXML($xml);
        }

        $xml->registerXPathNamespace('xmlns', 'http://mws.amazonaws.com/doc/2009-01-01/');

        $xpath_result = $xml->xpath('//xmlns:GetReportScheduleListResponse/xmlns:GetReportScheduleListResult/xmlns:ReportSchedule');

        if (Amazon::$debug_mode) {
            echo '<pre>GetReportScheduleListResponse:'.Amazon::LF;
            printf('%s(#%d): reportList result: %s', basename(__FILE__), __LINE__, nl2br(print_r($xpath_result, true)));
            echo '</pre>'.Amazon::LF;
        }
        self::$next_action = null;

        if (is_array($xpath_result) && !count($xpath_result)) {
            if ($xml instanceof SimpleXMLElement && isset($xml->GetReportScheduleListResult)) { // HasNext: doesn't matter the value
                $frequency_text = $this->l('Activation pending...');
                self::$next_action = self::ACTION_ACTIVATE;
                $pending = true;

                self::$messages[] = $message = $frequency_text;

                if (Amazon::$debug_mode) {
                    echo "<pre>Current Status:\n";
                    echo $message;
                    echo '</pre>'.Amazon::LF;
                }
                return(true);
            } else {
                if (Amazon::$debug_mode) {
                    echo "<pre>No Result\n";
                    echo '</pre>'.Amazon::LF;
                }
                return(false);
            }
        } elseif (is_array($xpath_result) && count($xpath_result)) {
            // the report is available, take the first one :
            $schedule_response = reset($xpath_result);

            if (Amazon::$debug_mode) {
                echo '<pre>Schedule:'.Amazon::LF;
                printf('getReportScheduleList() Error: %s'.Amazon::LF, print_r($schedule_response, true));
                echo "</pre>".Amazon::LF;
            }

            if ($schedule_response instanceof SimpleXMLElement) {
                switch ($schedule_response->Schedule) {
                    case '_30_MINUTES_':
                        $frequency_text = $this->l('30 minutes');
                        $pending = false;
                        break;
                    case '_1_HOUR_':
                        $frequency_text = $this->l('1 hour');
                        $pending = false;
                        break;
                    default:
                        $frequency_text = $this->l('Activation pending...');
                        self::$next_action = self::ACTION_ACTIVATE;
                        $pending = true;
                        break;

                }

                if ($pending == true) {
                    self::$messages[] = $message = $frequency_text;

                    if (Amazon::$debug_mode) {
                        echo "<pre>Current Status:\n";
                        echo $message;
                        echo '</pre>'.Amazon::LF;
                    }

                    return(true);
                } else {
                    self::$messages[] = $message = sprintf('%s: %s, %s: %s', $this->l('Frequency'), $frequency_text, $this->l('Next Report'), AmazonTools::displayDate(date('Y-m-d H:i:s', strtotime($schedule_response->ScheduledDate)), $this->id_lang, true));

                    if (Amazon::$debug_mode) {
                        echo "<pre>Current Status:\n";
                        echo $message;
                        echo '</pre>'.Amazon::LF;
                    }
                    return(false);
                }
            } else {
                $error = 'No Result';

                if (Amazon::$debug_mode) {
                    echo '<pre>'.Amazon::LF;
                    printf('getReportScheduleList() Error: %s'.Amazon::LF, $error);
                    echo "</pre>".Amazon::LF;
                }
                return(false);
            }
        }

        return (false);
    }


    public function manageReportSchedule()
    {
        $params = array();
        $params['Version'] = '2009-01-01';
        $params['Action'] = 'ManageReportSchedule';
        $params['ReportType'] = '_GET_ORDERS_DATA_';
        $params['Schedule'] = '_1_HOUR_';

        if (Amazon::$debug_mode) {
            echo '<pre>'.Amazon::LF;
            printf('manageReportSchedule()'.Amazon::LF);
            echo "</pre>".Amazon::LF;
        }

        $xml = $this->simpleCallWS($params, 'Reports');

        if (!$xml instanceof SimpleXMLElement or isset($xml->Error)) {
            die(sprintf('%s(#%d): ManageReportScheduleResultResponse Failed', basename(__FILE__), __LINE__));
        }

        if (Amazon::$debug_mode) {
            echo '<pre>'.Amazon::LF;
            printf('manageReportSchedule() - report'.Amazon::LF);
            echo "</pre>".Amazon::LF;
            echo $this->debugXML($xml);
        }

        $xml->registerXPathNamespace('xmlns', 'http://mws.amazonaws.com/doc/2009-01-01/');

        $xpath_result = $xml->xpath('//xmlns:ManageReportScheduleResponse/xmlns:ManageReportScheduleResult/xmlns:ReportSchedule');

        if (Amazon::$debug_mode) {
            echo '<pre>ManageReportScheduleResultResponse:'.Amazon::LF;
            printf('%s(#%d): result: %s', basename(__FILE__), __LINE__, nl2br(print_r($xpath_result, true)));
            echo '</pre>'.Amazon::LF;
        }
        self::$next_action = null;

        $next_schedule = null;

        if (is_array($xpath_result) && !count($xpath_result)) {
            if (Amazon::$debug_mode) {
                echo '<pre>'.Amazon::LF;
                printf('%s(#%d): manageReportSchedule(): Failed - %s', basename(__FILE__), __LINE__, 'No Result');
                echo "</pre>".Amazon::LF;
            }
            die;
        } elseif (is_array($xpath_result) && count($xpath_result)) {
            $pass = false;

            foreach ($xpath_result as $report_schedule) {
                if ($report_schedule instanceof SimpleXMLElement) {
                    if (Amazon::$debug_mode) {
                        echo '<pre>'.Amazon::LF;
                        printf('ReportSchedule - entry: %s'.Amazon::LF, print_r($report_schedule->asXML(), true));
                        echo "</pre>".Amazon::LF;
                    }
                    if ($report_schedule->Schedule == '_NEVER_') {
                        continue;
                    }
                }
                $next_schedule = $report_schedule->ScheduledDate;

                switch ($report_schedule->Schedule) {
                    case '_30_MINUTES_':
                        $frequency_text = $this->l('30 minutes');
                        $pass = true;
                        break;
                    case '_1_HOUR_':
                        $frequency_text = $this->l('1 hour');
                        $pass = true;
                        break;
                    default:
                        $frequency_text = $this->l('No schedule found, please try to reload the page...');
                        break;

                }
                break;
            }

            if ($pass == true) {
                self::$messages[] = $message = sprintf('%s: %s, %s: %s', $this->l('Frequency'), $frequency_text, $this->l('Next Schedule'), AmazonTools::displayDate(date('Y-m-d H:i:s', strtotime($next_schedule)), $this->id_lang, true));

                if (Amazon::$debug_mode) {
                    echo "<pre>Current Schedule:\n";
                    echo $message;
                    echo '</pre>'.Amazon::LF;
                }
                return(true);
            } else {
                if (Amazon::$debug_mode) {
                    echo '<pre>'.Amazon::LF;
                    printf('%s(#%d): manageReportSchedule(): Failed - %s', basename(__FILE__), __LINE__, 'No Result');
                    echo "</pre>".Amazon::LF;
                }

                die;
            }
        }

        return (false);
    }


    protected function debugXML($xml)
    {
        $dom = dom_import_simplexml($xml)->ownerDocument;
        $dom->formatOutput = true;

        $output = '<pre>';
        $output .= htmlspecialchars($dom->saveXML());
        $output .= '</pre>';

        return ($output);
    }
}