<?php
/**
 * 2007-2015 PrestaShop
 *
 * NOTICE OF LICENSE
 *
 * This source file is subject to the Academic Free License (AFL 3.0)
 * that is bundled with this package in the file LICENSE.txt.
 * It is also available through the world-wide-web at this URL:
 * http://opensource.org/licenses/afl-3.0.php
 * If you did not receive a copy of the license and are unable to
 * obtain it through the world-wide-web, please send an email
 * to license@prestashop.com so we can send you a copy immediately.
 *
 * DISCLAIMER
 *
 * Do not edit or add to this file if you wish to upgrade PrestaShop to newer
 * versions in the future. If you wish to customize PrestaShop for your
 * needs please refer to http://www.prestashop.com for more information.
 *
 *  @author    Needacart.com
 *  @copyright 2007-2015 PrestaShop SA
 *  @license   http://opensource.org/licenses/afl-3.0.php  Academic Free License (AFL 3.0)
 *  International Registered Trademark & Property of PrestaShop SA
 */

class NoCaptchaRecaptcha extends Module
{
	/**
	 * Probability of login attempts db table garbage cleanup
	 *
	 * @var int
	 */
	const GC_PROBABILITY = 10;

	/**
	 * Flag whether increment login attempts in destructor
	 *
	 * @var bool
	 */
	private $increment_login_attempts_in_destructor = false;

	/**
	 * Prestashop verion MAJOR.MINOR
	 *
	 * @var string
	 */
	private $is16;

	public function __construct()
	{
		$this->name = 'nocaptcharecaptcha';
		$this->tab = 'front_office_features';
		$this->version = '2.0.1';
		$this->author = 'needacart.com';

		parent::__construct();

		$this->displayName = $this->l('No CAPTCHA reCAPTCHA');
		$this->description = $this->l('Adds google reCAPTCHA to contact and register, login forms');

		$this->confirmUninstall = $this->l('Are you sure you want to uninstall this module?');
		$this->ps_versions_compliancy = array('min' => '1.5', 'max' => _PS_VERSION_);
		$this->is16 = _PS_VERSION_ >= 1.6;
		if (!$this->is16 )
			// <= 1.5.6 version test compatibility
			$this->ps_versions_compliancy = array('min' => '1.5', 'max' => _PS_VERSION_.'.1');
		else
			$this->ps_versions_compliancy = array('min' => '1.5', 'max' => _PS_VERSION_);
		$this->bootstrap = $this->is16;
		if (!Configuration::get('RECAPTCHA_SITE_KEY') || !Configuration::get('RECAPTCHA_SECRET_KEY'))
			$this->warning = $this->l('API keys is not properly set');
		$this->module_key = '9e8f4e46ace77d396c3306c40e4717b9';

		if ($this->isInstalled($this->name))
			//Cleanup login attempts database in ~ self::GC_PROBABILITY % requests
			if (mt_rand(1, 100) > (100 - self::GC_PROBABILITY))
				$this->cleanLoginTable();
	}

	public function install()
	{
		$sql = 'CREATE TABLE IF NOT EXISTS `'._DB_PREFIX_.$this->name.'_login_attempts` (
			`ip` int(10) unsigned NOT NULL,
			`attempts` tinyint(3) unsigned NOT NULL DEFAULT 1,
			`last_attempt` timestamp NOT NULL DEFAULT CURRENT_TIMESTAMP,
			PRIMARY KEY (`ip`),
			INDEX(`last_attempt`)
		) ENGINE='._MYSQL_ENGINE_.' DEFAULT CHARSET=utf8';
		if (!parent::install()
		|| !Configuration::updateValue('RECAPTCHA_SITE_KEY', '')
		|| !Configuration::updateValue('RECAPTCHA_SECRET_KEY', '')
		|| !Configuration::updateValue('RECAPTCHA_THEME', '')
		|| !Configuration::updateValue('RECAPTCHA_TYPE', '')
		|| !Configuration::updateValue('RECAPTCHA_CONTACT', false)
		|| !Configuration::updateValue('RECAPTCHA_LOGIN', false)
		|| !Configuration::updateValue('RECAPTCHA_LOGIN_ATTEMPTS', 3)
		|| !Configuration::updateValue('RECAPTCHA_REGISTER', false)
		|| !Db::getInstance()->execute($sql)
		|| !$this->registerHook('header')
		|| !$this->registerHook('createAccountTop'))
			return false;
		return true;
	}

	public function uninstall()
	{
		$sql = 'DROP TABLE IF EXISTS `'._DB_PREFIX_.$this->name.'_login_attempts`';
		if (!Configuration::deleteByName('RECAPTCHA_SITE_KEY')
		|| !Configuration::deleteByName('RECAPTCHA_SECRET_KEY')
		|| !Configuration::deleteByName('RECAPTCHA_THEME')
		|| !Configuration::deleteByName('RECAPTCHA_TYPE')
		|| !Configuration::deleteByName('RECAPTCHA_CONTACT')
		|| !Configuration::deleteByName('RECAPTCHA_LOGIN')
		|| !Configuration::deleteByName('RECAPTCHA_REGISTER')
		|| !Db::getInstance()->execute($sql)
		|| !parent::uninstall())
			return false;
		return true;
	}

	/**
	 * Hook to add reCAPTCHA javascript in the header tag
	 */
	public function hookHeader()
	{
		$page_name = Dispatcher::getInstance()->getController();
		$show_recaptcha = false;
		switch ($page_name)
		{
			case 'contact':
				$show_recaptcha = Configuration::get('RECAPTCHA_CONTACT');
				break;
			case 'authentication':
				if (Configuration::get('RECAPTCHA_LOGIN'))
				{
					$max_login_attempts = Configuration::get('RECAPTCHA_LOGIN_ATTEMPTS');
					if ($max_login_attempts == 0 || ($max_login_attempts > 0 && $this->getLoginAttempts() >= $max_login_attempts))
						$show_recaptcha = true;
				}
				break;
		}
		if ($show_recaptcha)
		{
			/* @var $smarty Smarty */
			$site_key = Configuration::get('RECAPTCHA_SITE_KEY');
			$smarty = $this->context->smarty;
			$smarty->assign(array(
				'include_template' => $page_name.($this->is16 ? '16' : ''),
				'error' => (!$site_key ? $this->l('Please set reCAPTCHA keys on the module configuration page!') : ''),
				'site_key' => $site_key,
				'theme' => Configuration::get('RECAPTCHA_THEME'),
				'type' => Configuration::get('RECAPTCHA_TYPE'),
				'lang_iso2' => $this->context->language->iso_code
			));
			return $this->display(__FILE__, 'header.tpl');
		}
	}

	/**
	 * Hook for for create aacount page
	 */
	public function hookCreateAccountTop()
	{
		if (Configuration::get('RECAPTCHA_REGISTER'))
		{
			/* @var $smarty Smarty */
			$site_key = Configuration::get('RECAPTCHA_SITE_KEY');
			$smarty = $this->context->smarty;
			$smarty->assign(array(
				'error' => (!$site_key ? $this->l('Please set reCAPTCHA keys on the module configuration page!') : ''),
				'site_key' => $site_key,
				'theme' => Configuration::get('RECAPTCHA_THEME'),
				'type' => Configuration::get('RECAPTCHA_TYPE'),
			));
			return $this->display(__FILE__, 'authentication_register'.($this->is16 ? '16' : '').'.tpl');
		}
	}

	/**
	 * Module configuration page
	 */
	public function getContent()
	{
		$html = '';
		if (Tools::isSubmit('submitModule'))
		{
			Configuration::updateValue('RECAPTCHA_SITE_KEY', Tools::getValue('RECAPTCHA_SITE_KEY'));
			Configuration::updateValue('RECAPTCHA_SECRET_KEY', Tools::getValue('RECAPTCHA_SECRET_KEY'));
			Configuration::updateValue('RECAPTCHA_THEME', Tools::getValue('RECAPTCHA_THEME'));
			Configuration::updateValue('RECAPTCHA_TYPE', Tools::getValue('RECAPTCHA_TYPE'));
			Configuration::updateValue('RECAPTCHA_CONTACT', (bool)Tools::getValue('RECAPTCHA_CONTACT'));
			Configuration::updateValue('RECAPTCHA_LOGIN', (bool)Tools::getValue('RECAPTCHA_LOGIN'));
			Configuration::updateValue('RECAPTCHA_LOGIN_ATTEMPTS', (int)Tools::getValue('RECAPTCHA_LOGIN_ATTEMPTS'));
			Configuration::updateValue('RECAPTCHA_REGISTER', (bool)Tools::getValue('RECAPTCHA_REGISTER'));
			$html .= $this->displayConfirmation($this->l('Configuration updated'));
		}
		$html .= $this->is16 ? $this->getContent16() : $this->getContent15();
		return $html;
	}

	/**
	 * Show pretty 1.6 style form
	 */
	private function getContent16()
	{
		$fields_form = array(
			'form' => array(
				'legend' => array(
					'title' => $this->l('reCAPTCHA by needacart.com'),
				),
				'input' => array(
					array(
						'type' => 'text',
						'label' => $this->l('Site key'),
						'name' => 'RECAPTCHA_SITE_KEY',
					),
					array(
						'type' => 'text',
						'label' => $this->l('Secret key'),
						'name' => 'RECAPTCHA_SECRET_KEY',
						'desc' => $this->l('To get reCAPTCHA keys follow this ').' <a target="_blank" href="https://www.google.com/recaptcha/admin">'.
							$this->l('link').'</a>',
					),
					array(
						'type' => 'select',
						'label' => $this->l('reCAPTCHA theme'),
						'name' => 'RECAPTCHA_THEME',
						'options' => array(
							'query' => array(
								array(
									'id' => 'light',
									'name' => $this->l('light')
								),
								array(
									'id' => 'dark',
									'name' => $this->l('dark')
								)
							),
							'id' => 'id',
							'name' => 'name',
						)
					),
					array(
						'type' => 'select',
						'label' => $this->l('reCAPTCHA type'),
						'name' => 'RECAPTCHA_TYPE',
						'options' => array(
							'query' => array(
								array(
									'id' => 'image',
									'name' => $this->l('image')
								),
								array(
									'id' => 'audio',
									'name' => $this->l('audio')
								),
							),
							'id' => 'id',
							'name' => 'name',
						)
					),
					array(
						'type' => 'switch',
						'label' => $this->l('Enable on contact form'),
						'name' => 'RECAPTCHA_CONTACT',
						'values' => array(
							array(
								'id' => 'active_on',
								'value' => 1,
								'label' => $this->l('Enabled')
							),
							array(
								'id' => 'active_off',
								'value' => 0,
								'label' => $this->l('Disabled')
							)
						),
					),
					array(
						'type' => 'switch',
						'label' => $this->l('Enable on login form'),
						'name' => 'RECAPTCHA_LOGIN',
						'values' => array(
							array(
								'id' => 'active_on',
								'value' => 1,
								'label' => $this->l('Enabled')
							),
							array(
								'id' => 'active_off',
								'value' => 0,
								'label' => $this->l('Disabled')
							)
						),
					),
					array(
						'type' => 'text',
						'label' => $this->l('Show after login attempts (0 - always)'),
						'name' => 'RECAPTCHA_LOGIN_ATTEMPTS',
						'class' => 'fixed-width-xs'
					),
					array(
						'type' => 'switch',
						'label' => $this->l('Enable on register form'),
						'name' => 'RECAPTCHA_REGISTER',
						'values' => array(
							array(
								'id' => 'active_on',
								'value' => 1,
								'label' => $this->l('Enabled')
							),
							array(
								'id' => 'active_off',
								'value' => 0,
								'label' => $this->l('Disabled')
							)
						),
					),
				),
				'submit' => array(
					'title' => $this->l('Save'),
				)
			),
		);

		$helper = new HelperForm();
		$helper->show_toolbar = false;
		$helper->table = $this->table;
		$lang = new Language((int)Configuration::get('PS_LANG_DEFAULT'));
		$helper->default_form_language = $lang->id;
		$helper->allow_employee_form_lang = Configuration::get('PS_BO_ALLOW_EMPLOYEE_FORM_LANG') ? Configuration::get('PS_BO_ALLOW_EMPLOYEE_FORM_LANG') : 0;
		$this->fields_form = array();
		$helper->identifier = $this->identifier;
		$helper->submit_action = 'submitModule';
		$helper->currentIndex = $this->context->link->getAdminLink('AdminModules', false).'&configure='.$this->name.'&tab_module='.$this->tab.
			'&module_name='.$this->name;
		$helper->token = Tools::getAdminTokenLite('AdminModules');
		$helper->tpl_vars = array(
			'fields_value' => $this->getConfigFieldsValues(),
			'languages' => $this->context->controller->getLanguages(),
			'id_language' => $this->context->language->id
		);
		return $helper->generateForm(array($fields_form));
	}

	public function getConfigFieldsValues()
	{
		return array(
			'RECAPTCHA_SITE_KEY' => Tools::getValue('RECAPTCHA_SITE_KEY', Configuration::get('RECAPTCHA_SITE_KEY')),
			'RECAPTCHA_SECRET_KEY' => Tools::getValue('RECAPTCHA_SECRET_KEY', Configuration::get('RECAPTCHA_SECRET_KEY')),
			'RECAPTCHA_THEME' => Tools::getValue('RECAPTCHA_THEME', Configuration::get('RECAPTCHA_THEME')),
			'RECAPTCHA_TYPE' => Tools::getValue('RECAPTCHA_TYPE', Configuration::get('RECAPTCHA_TYPE')),
			'RECAPTCHA_CONTACT' => Tools::getValue('RECAPTCHA_CONTACT', Configuration::get('RECAPTCHA_CONTACT')),
			'RECAPTCHA_LOGIN' => Tools::getValue('RECAPTCHA_LOGIN', Configuration::get('RECAPTCHA_LOGIN')),
			'RECAPTCHA_LOGIN_ATTEMPTS' => Tools::getValue('RECAPTCHA_LOGIN_ATTEMPTS', Configuration::get('RECAPTCHA_LOGIN_ATTEMPTS')),
			'RECAPTCHA_REGISTER' => Tools::getValue('RECAPTCHA_REGISTER', Configuration::get('RECAPTCHA_REGISTER')),
		);
	}

	/**
	 * 1.5 style form
	 */
	private function getContent15()
	{
		return '<h2>'.$this->displayName.'</h2>
				<form action="'.Tools::htmlentitiesutf8($_SERVER['REQUEST_URI']).'" method="post">
				<fieldset>
					<legend>reCAPTCHA by needacart.com</legend>
					<p><label for="RECAPTCHA_SITE_KEY">'.$this->l('Site key').'</label>
					<input type="text" name="RECAPTCHA_SITE_KEY" style="width: 300px;" value="'.
						Tools::safeOutput(Configuration::get('RECAPTCHA_SITE_KEY')).'" /></p>
					<p><label for="RECAPTCHA_SECRET_KEY">'.$this->l('Secret key').'</label>
					<input type="text" name="RECAPTCHA_SECRET_KEY" style="width: 300px;" value="'.
						Tools::safeOutput(Configuration::get('RECAPTCHA_SECRET_KEY')).'" /></p>
					<p>'.$this->l('To get reCAPTCHA keys follow this ').' <a target="_blank" href="https://www.google.com/recaptcha/admin">'.
						$this->l('link').'</a></p>
					<p><label for="RECAPTCHA_THEME">'.$this->l('reCAPTCHA theme name').'</label>
					<select name="RECAPTCHA_THEME">
						<option value="light"'.(Configuration::get('RECAPTCHA_THEME') == 'light' ? ' selected="selected"' : '').'>'.$this->l('light').'</option>
						<option value="dark"'.(Configuration::get('RECAPTCHA_THEME') == 'dark' ? ' selected="selected"' : '').'>'.$this->l('dark').'</option>
					</select></p>
					<p><label for="RECAPTCHA_TYPE">'.$this->l('reCAPTCHA type').'</label>
					<select name="RECAPTCHA_TYPE">
						<option value="image"'.(Configuration::get('RECAPTCHA_TYPE') == 'image' ? ' selected="selected"' : '').'>'.$this->l('image').'</option>
						<option value="audio"'.(Configuration::get('RECAPTCHA_TYPE') == 'audio' ? ' selected="selected"' : '').'>'.$this->l('audio').'</option>
					</select></p>
					<p><label for="RECAPTCHA_CONTACT">'.$this->l('Enable on contact form').'</label>
					<input type="checkbox" name="RECAPTCHA_CONTACT" '.((Configuration::get('RECAPTCHA_CONTACT')) ? ' checked=""': '').'/></p>
					<p><label for="RECAPTCHA_LOGIN">'.$this->l('Enable on login form').'</label>
					<input type="checkbox" name="RECAPTCHA_LOGIN" '.((Configuration::get('RECAPTCHA_LOGIN')) ? ' checked=""': '').'/></p>
					<p><label for="RECAPTCHA_LOGIN_ATTEMPTS">'.$this->l('Show after login attempts (0 - always)').'</label>
					<input type="text" name="RECAPTCHA_LOGIN_ATTEMPTS" value="'.Tools::safeOutput(Configuration::get('RECAPTCHA_LOGIN_ATTEMPTS')).'" /></p>
					<p><label for="RECAPTCHA_REGISTER">'.$this->l('Enable on register form').'</label>
					<input type="checkbox" name="RECAPTCHA_REGISTER" '.((Configuration::get('RECAPTCHA_REGISTER')) ? ' checked=""': '').'/></p>
					<div class="margin-form">
						<input type="submit" name="submitModule" value="'.$this->l('Update settings').'" class="button" /></center>
					</div>
				</fieldset>
			</form>';
	}

	/**
	 * Validate a reCAPTCHA
	 *
	 * @return ReCaptchaResponse
	 */
	public function check()
	{
		//include google's library https://code.google.com/p/recaptcha/
		require_once(dirname(__FILE__).'/lib/recaptchalib.php');
		$re_captcha = new ReCaptcha(Configuration::get('RECAPTCHA_SECRET_KEY'));
		//Reverse proxy support with Tools::getRemoteAddr()
		$response = $re_captcha->verifyResponse(Tools::getRemoteAddr(), Tools::getValue('g-recaptcha-response'));
		return $response;
	}

	/**
	 * Get default error text (translated)
	 */
	public function errorText()
	{
		return $this->l('Invalid reCAPTCHA');
	}

	/**
	 * Get login attempts for current IP
	 *
	 * @return int
	 */
	public function getLoginAttempts()
	{
		return (int)Db::getInstance()->getValue('SELECT `attempts`
			FROM `'._DB_PREFIX_.$this->name."_login_attempts`
			WHERE `ip` = '".(int)ip2long(Tools::getRemoteAddr())."'");
	}

	/**`'._DB_PREFIX_.$this->name.'_login_attempts`
	 * Increment login attempts
	 *
	 * @param bool $in_destructor do not increment immediately, do this in destructor
	 * @return void|boolean
	 */
	public function incrementLoginAttempts($in_destructor = false)
	{
		if ($in_destructor)
		{
			$this->increment_login_attempts_in_destructor = true;
			return;
		}
		return Db::getInstance()->execute('INSERT INTO `'._DB_PREFIX_.$this->name."_login_attempts` (`ip`)
			VALUES ('".(int)ip2long(Tools::getRemoteAddr())."')
			ON DUPLICATE KEY UPDATE `attempts` = `attempts` + 1");
	}

	/**
	 * Increment login attempts
	 *
	 * @return void|boolean
	 */
	public function deleteLoginAttempts()
	{
		return Db::getInstance()->execute('DELETE FROM `'._DB_PREFIX_.$this->name."_login_attempts`
			WHERE `ip` = '".(int)ip2long(Tools::getRemoteAddr())."'");
	}

	/**
	 * Prestashop handles AJAX responses strange... using die() in the controller...
	 */
	public function __destruct()
	{
		if ($this->increment_login_attempts_in_destructor)
			$this->incrementLoginAttempts();
	}

	/**
	 * Cleanup table with invalid login attempts
	 *
	 * @return boolean
	 */
	private function cleanLoginTable()
	{
		return Db::getInstance()->execute('DELETE FROM `'._DB_PREFIX_.$this->name.'_login_attempts`
					WHERE `last_attempt` <= NOW() - INTERVAL 1 DAY');
	}

}