<?php
/**
 * Scheduled export configuration use by the module export_catalog
 *
 * @category  Prestashop
 * @category  Module
 * @author    Samdha <contact@samdha.net>
 * @copyright Samdha
 * @license   commercial license see license.txt
 */

class Samdha_ExportCatalog_Export extends Samdha_ExportCatalog_Configuration
{
	protected $extension = '.export';

	/**
	 * load configuration from a file
	 * @param  string $filename file to load
	 * @return boolean           true if succeeded
	 */
	public function loadFromFile($filename)
	{
		$result = parent::loadFromFile($filename);
		if ($result
			&& (
				!isset($this->datas['days'])
				|| !is_array($this->datas['days'])
			))
			$this->datas['days'] = array();
		return $result;
	}

	/**
	 * set default configuration
	 * @return void
	 */
	protected function initDefault()
	{
		$this->name = ''; // export name
		$this->datas = array(
			'model' => '', // associated export model
			'id_employee' => '0', // employee that will receive the file by mail
			'folder' => '', // directory where the file will be copied
			'days' => array(0,1,2,3,4,5,6), // day when run the export
			'hour' => '0', // hour when run the export
			'minute' => '0', // minute when run the export
		);
	}

	/**
	 * get when the export should be run next time
	 * @return timestamp
	 */
	public function getNextRun()
	{
		if (empty($this->datas['days']))
			return PHP_INT_MAX;

		$day = (int)date('w');
		$minutes = 60 * date('G') + date('i');
		$wanted_minutes = 60 * $this->datas['hour'] + $this->datas['minute'];
		for ($i = 0; $i <= 7; $i++)
		{
			if (in_array(($day + $i) % 7, $this->datas['days'])
				&& ($minutes <= $wanted_minutes))
				break;
			$minutes = 0;
		}
		return mktime($this->datas['hour'], $this->datas['minute'], 0, date('n'), date('j') + $i);
	}

	/**
	 * get associated export model
	 * @return object Samdha_ExportCatalog_Model
	 */
	public function getModel()
	{
		return new Samdha_ExportCatalog_Model(
			$this->directory,
			$this->module,
			$this->datas['model']
		);
	}

	/**
	 * Do the export
	 * the file to export has been generated before by the model
	 * it will be deleted by this methods
	 *
	 * @param  string $filename file to export
	 * @return void
	 */
	public function export($filename)
	{
		$model = $this->getModel();
		$model->datas['last_run'] = time();
		$model->save();
		$model->restoreContext();

		// send by mail if needed
		if ((int)$this->datas['id_employee'])
		{
			$employee = new Employee((int)$this->datas['id_employee']);

			$file_attachment = array(
				'content' => $this->module->samdha_tools->fileGetContents($filename),
				'name' => strftime($model->datas['filename']),
				'mime' => 'text/csv'
			);

			Mail::Send(
				Configuration::get('PS_LANG_DEFAULT'),
				$this->module->name,
				$this->module->name,
				array(),
				$employee->email,
				$employee->firstname.' '.$employee->lastname,
				null,
				null,
				$file_attachment,
				null,
				$this->module->mail_path
			);
		}
		// save in folder if needed
		if ($this->datas['folder'])
		{
			$new_name = _PS_ROOT_DIR_.$this->datas['folder'].strftime($model->datas['filename']);
			if (rename($filename, $new_name))
				@chmod($new_name, 0755);
		}
		// deleted the file if still exists
		if (file_exists($filename))
			@unlink($filename);
	}
}
