<?php
/**
 * NOTICE OF LICENSE
 *
 * This source file is subject to a commercial license from Common-Services Co., Ltd.
 * Use, copy, modification or distribution of this source file without written
 * license agreement from the SARL SMC is strictly forbidden.
 * In order to obtain a license, please contact us: contact@common-services.com
 * ...........................................................................
 * INFORMATION SUR LA LICENCE D'UTILISATION
 *
 * L'utilisation de ce fichier source est soumise a une licence commerciale
 * concedee par la societe Common-Services Co., Ltd.
 * Toute utilisation, reproduction, modification ou distribution du present
 * fichier source sans contrat de licence ecrit de la part de la Common-Services Co. Ltd. est
 * expressement interdite.
 * Pour obtenir une licence, veuillez contacter Common-Services Co., Ltd. a l'adresse: contact@common-services.com
 *
 * @author    Olivier B.
 * @copyright Copyright (c) 2011-2018 Common Services Co Ltd - 90/25 Sukhumvit 81 - 10260 Bangkok - Thailand
 * @license   Commercial license
 * @package   Amazon Market Place
 * Support by mail:  support.amazon@common-services.com
*/

require_once(dirname(__FILE__).'/env.php');
require_once(dirname(__FILE__).'/../amazon.php');

require_once(dirname(__FILE__).'/../classes/amazon.batch.class.php');
require_once(dirname(__FILE__).'/../classes/amazon.webservice.class.php');

class AmazonProductReport extends Amazon
{
    public static $errors   = array();
    public static $warnings = array();
    public static $messages = array();
    public static $report   = array();

    private $_debug = false;

    public function __construct()
    {
        parent::__construct();

        if ((int)Tools::getValue('id_lang')) {
            $this->id_lang = (int)Tools::getValue('id_lang');
        }

        AmazonContext::restore($this->context);
    }

    public function doIt()
    {
        ob_start();

        $id_lang = (int)Tools::getValue('amazon_lang');

        $callback = Tools::getValue('callback');
        $action = (string)Tools::getValue('action');
        $cr = "<br />\n";

        $this->_debug = (bool)Configuration::get('AMAZON_DEBUG_MODE');

        if (Tools::getValue('debug')) {
            $this->_debug = true;
        }

        if ($this->_debug) {
            @ini_set('display_errors', 'on');
            @error_reporting(E_ALL | E_STRICT);
        }
        // Init
        //
        $amazon = AmazonTools::selectPlatforms($id_lang, $this->_debug);

        if ($this->_debug) {
            echo nl2br(print_r($amazon['auth'], true).print_r($amazon['params'], true).print_r($amazon['platforms'], true));
        }

        if (!($this->_amazonApi = new AmazonWebService($amazon['auth'], $amazon['params'], $amazon['platforms'], $this->_debug))) {
            echo $this->l('Unable to login').$cr;
            die;
        }

        switch ($action) {
            case 'display-statistics':
                $this->displayStatistics();
                break;
            case 'purge':
                $this->purge();
                break;
            case 'list-reports':
                $this->listReports();
                break;
            case 'one-report':
                $html = '';
                $pass = false;


                $reportid = Tools::getValue('reportid');
                $type = Tools::getValue('type');

                // Submission Report
                //
                $response = $this->_amazonApi->getFeedSubmissionResult($reportid);

                if (isset($response->Error->Code) && (string)$response->Error->Code == 'FeedProcessingResultNotReady') {
                    $pass = true;

                    if ($this->ps16x) {
                        $class_error = 'alert alert-warning';
                    } else {
                        $class_error = 'warn';
                    }

                    $html .= '<div class="'.$class_error.'">'.$this->l('Report is not available yet, please wait few minutes').'</div>';
                } elseif ($response instanceof SimpleXMLElement) {
                    $html .= $this->displayGetFeedSubmissionResult($response, sprintf('Feed - %s: %s %s', AmazonTools::ucfirst($type), $this->l('ID'), $reportid));
                    $pass = true;
                }

                // Debug Messages
                //
                if (!$pass) {
                    echo $this->l('No available reports').$cr;
                }

                $output = '<div class="report-debug-div">';
                $output .= ob_get_clean();
                $output .= '</div>';

                echo $html;

                echo $output;

                break;
        }
    }

    public function l($string, $specific = false, $id_lang = null)
    {
        return (parent::l($string, basename(__FILE__, '.php'), $id_lang));
    }

    public function displayStatistics()
    {
        $i = 0;
        $full_batch_list = array();
        $html = null;
        $stats = AmazonProduct::marketplaceCountProducts();

        $actives = unserialize(AmazonTools::decode(Configuration::get('AMAZON_ACTIVE')));

        $statistics = array();

        foreach (array('session_products', 'session_offers', 'session_repricing', 'session_status', 'session_import') as $batch_type) {
            $batches = new AmazonBatches($batch_type);
            $batches_list = $batches->load();

            switch ($batch_type) {
                case 'session_products':
                    $type = $this->l('Products');
                    break;
                case 'session_offers':
                    $type = $this->l('Offers');
                    break;
                case 'session_repricing':
                    $type = $this->l('Repricing');
                    break;
                case 'session_status':
                    $type = $this->l('Statuses (Orders)');
                    break;
                case 'session_import':
                    $type = $this->l('Orders (Import)');
                    break;
                default:
                    $type = $this->l('Other');
                    break;
            }

            foreach ($batches_list as $batch) {
                $index = $batch->timestart.'.'.$i;

                $statistics[$index] = $batch->format();
                $statistics[$index]['records'] = $batch->created + $batch->updated + $batch->deleted;
                $statistics[$index]['mode'] = $type;

                $i++;
            }
        }
        krsort($statistics);

        $html .= '
            <p>
                <b> '.$this->l('Statistics').'</b>: '.sprintf($this->l('There are %d synchronizable products and %d combinations in %d languages'), $stats['products'], $stats['attributes'], count($actives)).'
                <br /><br />
                <b>'.$this->l('Latest Updates').'</b>:

                <table style="width:100%">
                <tr style="font-style:italic"><td width="40%">'.$this->l('Action').'</td><td>'.$this->l('Type').'</td><td>'.$this->l('Data').'</td><td>'.$this->l('Date/Time').'</td><td>'.$this->l('Items').'</td><td>'.$this->l('Duration').'</td></tr>';

        $count = 0;
        foreach ($statistics as $statistic) {
            $count++;

            $html .= '<tr><td>Amazon.'.$statistic['region'].'</td><td>'.$statistic['type'].'</td><td>'.$statistic['mode'].'</td><td>'.$statistic['timestart'].'</td><td>'.$statistic['records'].'</td><td>'.$statistic['duration'].'</td></tr>';
        }

        $html .= '
                </table>';

        if (!$count) {
            $html .= '<span style="color:brown;font-weight:bold;">'.$this->l('No recorded updates').'</span>';
        }

        echo $html;
    }

    private function displayGetFeedSubmissionResult($obj, $title)
    {
        $html = '';
        $errors = '';

        if (isset($obj->Message->ProcessingReport->Result->ResultCode) && ($obj->Message->ProcessingReport->Result->ResultCode == 'Warning' || $obj->Message->ProcessingReport->Result->ResultCode == 'Error')) {
            if (is_object($obj->Message->ProcessingReport->Result)) {
                foreach ($obj->Message->ProcessingReport->Result as $result) {
                    if (isset($result->AdditionalInfo->SKU)) {
                        $additionnal_information = sprintf('SKU: %s - ', $result->AdditionalInfo->SKU);
                    } else {
                        $additionnal_information = '';
                    }

                    $errors .= nl2br(sprintf("Message: %d %s %d: %s%s\n\n", $result->MessageID, $result->ResultCode, $result->ResultMessageCode, $additionnal_information, $result->ResultDescription));
                }
            } else {
                if (isset($result->AdditionalInfo->SKU)) {
                    $additionnal_information = sprintf('SKU: %s - ', $result->AdditionalInfo->SKU);
                } else {
                    $additionnal_information = '';
                }

                $errors .= nl2br(sprintf("Message: %d %s %d: %s%s\n\n", $result->MessageID, $obj->Message->ProcessingReport->Result->ResultCode, $obj->Message->ProcessingReport->Result->ResultMessageCode, $additionnal_information, $obj->Message->ProcessingReport->Result->ResultDescription));
            }
        } elseif (isset($obj->Message->ProcessingReport->StatusCode) && $obj->Message->ProcessingReport->StatusCode == 'Complete') {
            $errors = '';
        } else {
            $errors .= nl2br(print_r($obj, true));
        }

        $html .= '
              <span class="submission-result-title">'.$title.'</span><br />';

        if (isset($obj->Message) && isset($obj->Message->ProcessingReport->Summary->ProcessingSummary)) {
            $summary = $obj->Message->ProcessingReport->Summary->ProcessingSummary;

            $html .= '
              <table class="submission-results-table">
                  <thead>
                      <tr>
                          <th>'.$this->l('Entry').'</th>
                          <th>'.$this->l('Result').'</th>
                      </tr>
                  </thead>
                  <tbody>
                     <tr>
                          <td  class="col1">'.$this->l('Entries Processed').'</td>
                          <td style="font-weight:bold;">'.(string)$summary->MessagesProcessed.'</td>
                     <tr>
                     </tr>
                          <td  class="col1">'.$this->l('Entries processed successfully').'</td>
                          <td style="color:#1D910F;font-weight:bold;">'.(string)$summary->MessagesSuccessful.'</td>
                     <tr>
                     </tr>
                          <td class="col1">'.$this->l('Entries with Error').'</td>
                          <td style="color:#FD0000;font-weight:bold;">'.(string)$summary->MessagesWithError.'</td>
                     <tr>
                     </tr>
                          <td class="col1">'.$this->l('Entries with Warning').'</td>
                          <td style="color:#FFA300;font-weight:bold;">'.(string)$summary->MessagesWithWarning.'</td>
                     </tr>
                     </tr>
                          <td class="col1">'.$this->l('Messages Logs').'</td>
                          <td>'.$errors.'</td>
                     </tr>
                  </tbody>
              </table>
              ';
        } elseif (isset($obj->Message) && isset($obj->Message->ProcessingReport->ProcessingSummary)) {
            $summary = $obj->Message->ProcessingReport->ProcessingSummary;

            $html .= '
              <table class="submission-results-table">
                  <thead>
                      <tr>
                          <th>'.$this->l('Entry').'</th>
                          <th>'.$this->l('Result').'</th>
                      </tr>
                  </thead>
                  <tbody>
                     <tr>
                          <td  class="col1">'.$this->l('Entries Processed').'</td>
                          <td style="font-weight:bold;">'.(string)$summary->MessagesProcessed.'</td>
                     <tr>
                     </tr>
                          <td  class="col1">'.$this->l('Entries processed successfully').'</td>
                          <td style="color:#1D910F;font-weight:bold;">'.(string)$summary->MessagesSuccessful.'</td>
                     <tr>
                     </tr>
                          <td class="col1">'.$this->l('Entries with Error').'</td>
                          <td style="color:#FD0000;font-weight:bold;">'.(string)$summary->MessagesWithError.'</td>
                     <tr>
                     </tr>
                          <td class="col1">'.$this->l('Entries with Warning').'</td>
                          <td style="color:#FFA300;font-weight:bold;">'.(string)$summary->MessagesWithWarning.'</td>
                     </tr>
                     </tr>
                          <td class="col1">'.$this->l('Messages Logs').'</td>
                          <td>'.$errors.'</td>
                     </tr>
                  </tbody>
              </table>
              ';
        } elseif (isset($obj->Error)) {
            /*
              SimpleXMLElement Object
              (
              [Error] => SimpleXMLElement Object
              (
              [Type] => Sender
              [Code] => FeedProcessingResultNotReady
              [Message] => Feed Submission Result is not ready for Feed Submission Id: 3899441890
              [Detail] => SimpleXMLElement Object
              (
              )
             */
            $error = $obj->Error->Message;

            $html .= '
              <table class="submission-results-table">
                  <thead>
                      <tr>
                          <th>'.$this->l('Entry').'</th>
                          <th>'.$this->l('Result').'</th>
                      </tr>
                  </thead>
                     </tr>
                          <td class="col1">'.$this->l('Error').'</td>
                          <td style="color:#FFA300;font-weight:bold;">'.(string)$obj->Error->Code.'</td>
                     </tr>
                  </body>';

            $html .= '
                     </tr>
                          <td class="col1">'.$this->l('Messages Logs').'</td>
                          <td>'.$error.'</td>
                     </tr>';

            // Display an interpretation of this usual case
            //
            if ((string)$obj->Error->Code == 'FeedProcessingResultNotReady') {
                $html .= '
                         </tr>
                              <td>&nbsp;</td>
                              <td class="interpretation"><b>'.$this->l('*** This is not an error *** Please DO NOT contact the support :').'</b><br />'.$this->l('Amazon is currently processing your request, please wait a while, the report will be available soon.').'</td>
                         </tr>';
            }

            $html .= '
                  </tbody>
              </table>';
        } else {
            $html .= nl2br(print_r($obj, true));
        }

        $html .= '<br /><hr /><br />';

        return ($html);
    }

    public function purge()
    {
        foreach (array(
                     'batch_products_cron',
                     'batch_products',
                     'batch_offers_cron',
                     'batch_offers',
                     'batch_repricing',
                     'batch_status',
                     'session_products',
                     'session_offers',
                     'session_repricing',
                     'session_status'
                 ) as $batch_type) {
            $batches = new AmazonBatches($batch_type);
            $batches->deleteKey();
        }
        die($this->l('Batches entries have been deleted'));
    }

    public function listReports()
    {
        $i = 0;
        ob_start();
        $full_batch_list = array();

        foreach (array(
                     'batch_products_cron',
                     'batch_products',
                     'batch_offers_cron',
                     'batch_offers',
                     'batch_repricing',
                     'batch_status',
                     'batch_acknowledge'
                 ) as $batch_type) {
            $batches = new AmazonBatches($batch_type);
            $batches_list = $batches->load();

            $full_batch_list = array_merge($full_batch_list, $batches_list);
        }

        if (is_array($full_batch_list) && count($full_batch_list)) {
            foreach ($full_batch_list as $batch) {
                $index = sprintf('%016s.%03s', $batch->timestart, $i);

                self::$report[$index] = $batch->format();
                self::$report[$index]['records'] = $batch->created + $batch->updated + $batch->deleted;

                $i++;
            }
            krsort(self::$report);
        }

        $result = trim(ob_get_clean());

        if (!empty($result)) {
            self::$warnings[] = trim($result);
        }

        $json = Tools::jsonEncode(array(
            'count' => count(self::$report),
            'reports' => self::$report,
            'error' => (count(self::$errors) ? true : false),
            'errors' => self::$errors,
            'warning' => (count(self::$warnings) ? true : false),
            'warnings' => self::$warnings,
            'message' => count(self::$messages),
            'messages' => self::$messages
        ));

        if (($callback = Tools::getValue('callback'))) {
            // jquery

            echo $callback.'('.$json.')';
        } else {
            echo "<pre>\n";
            echo $json;
            echo "<pre>\n";
        }
    }
}

$apr = new AmazonProductReport();

$apr->doIt();
